<?php
// Prevents the script from stopping when a user aborts the connection
ignore_user_abort(true);
// Removes the maximum time limit to execute the script
set_time_limit(0);

// Including necessary files for the script execution
require_once("../apis/pbx.php");
require_once("../model/query.php");
require_once("../inc/helper.php"); // Utility for database backup


$lockFile = "/tmp/summa_call_recording_fetch.lock";
$maxRunningInstances = 3; // Set the maximum allowed running instances

// Function to check if a given PID is running
function isPidRunning($pid) {
    // Check if the process is running in UNIX
    $result = shell_exec(sprintf("ps %d", $pid));
    if (count(preg_split("/\n/", $result)) > 2) {
        return true; // PID is running
    }
    return false; // PID is not running
}

// Function to get the running PIDs from the lock file
function getRunningPids($lockFile) {
    if (file_exists($lockFile)) {
        $pidData = json_decode(file_get_contents($lockFile), true);
        if (!is_array($pidData)) $pidData = []; // Ensure returned data is an array
    } else {
        $pidData = [];
    }
    // Clean up any PIDs that are no longer running
    foreach ($pidData as $index => $pid) {
        if (!isPidRunning($pid)) {
            unset($pidData[$index]);
        }
    }
    return array_values($pidData); // Re-index the array
}

// Check the number of running instances and update the lock file accordingly
$runningPids = getRunningPids($lockFile);
if (count($runningPids) >= $maxRunningInstances) {
    echo "The maximum number of instances ($maxRunningInstances) are already running. Exiting...\n";
    exit;
} else {
    echo "Starting new instance. Currently running instances: " . count($runningPids) . "\n";
    // Add the current PID to the list of running PIDs
    $runningPids[] = getmypid();
    // Save the updated list back to the lock file
    file_put_contents($lockFile, json_encode($runningPids));
}

// Ensure the lock file is updated on script exit to remove the current PID
register_shutdown_function(function() use ($lockFile) {
    $runningPids = getRunningPids($lockFile);
    // Remove the current PID from the list and save it
    if (($key = array_search(getmypid(), $runningPids)) !== false) {
        unset($runningPids[$key]);
    }
    file_put_contents($lockFile, json_encode(array_values($runningPids))); // Re-index array before saving
});

// Initialize classes for database operations, PBX API calls, and helper functions
$bu = new querys();
$pbx = new Pbx();
$hp = new Helper();

// Define the log file path
$logFile = "./log_summa_call_recording_fetch.log";


function addLog($message, $logFile, $type = 'INFO') {
    $timeStamp = date("Y-m-d H:i:s");
    $logEntry = sprintf("[%s] [%s] %s\n", $timeStamp, $type, $message);
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

// Main loop control variable
$seconds = 1;
do {
    // Fetch call details for recording
    $crf = $bu->summaCallDetailsForRecordingFetch();
    // Convert the $crf array to a JSON string for logging
	$crfJson = json_encode($crf);
	// Log the $crf values
	//addLog("Fetched call recording details: " . $crfJson, $logFile);

    // Log the fetch attempt with timestamp
    addLog("Fetched call recording details. Count: " . sizeof($crf), $logFile);

    // Check if records are available
    if (sizeof($crf)) {
        
        foreach ($crf as $item) {
            $bu->summaCallDetailsForRecordingFetchUpdateUdt($item['cdr_id']);//for avoiding concurrency
            
            // Attempt to fetch recordings for each item
            //$response = $pbx->getPbxUserCallRecording($item['domain'], $item['accountid'], $item['orig_callid'], $item['term_callid']);
            //$cr = json_decode($response);
	    // Enhanced API call logging example
		$response = $pbx->getPbxUserCallRecording($item['domain'], $item['accountid'], $item['orig_callid'], $item['term_callid']);
		$cr = json_decode($response);

		if ($cr === null) {
		    addLog("Failed to decode JSON response from getPbxUserCallRecording API. Response: " . $response, $logFile, 'ERROR');
		} else {
		    addLog("getPbxUserCallRecording API called successfully for accountid: " . $item['accountid'], $logFile);
		    if (!isset($cr->data->recording)) {
		        addLog("No recording found in API response for accountid: " . $item['accountid'], $logFile, 'WARNING');
		    }
		}
		addLog("API Request: getPbxUserCallRecording with domain: {$item['domain']}, accountid: {$item['accountid']}, orig_callid: {$item['orig_callid']}, term_callid: {$item['term_callid']}", $logFile);
		// Log API response status and data
		if(isset($http_response_header)){
		    list($version, $status_code, $msg) = explode(' ', $http_response_header[0], 3);
		    addLog("API Response Status: $status_code $msg", $logFile);
		    addLog("API Response Body: " . substr($response, 0, 400) . " [...]", $logFile); // Log first 400 chars for brevity
		}

          
            // Check if recording is available in the response
            if (isset($cr->data->recording)) {
                
                // Choose the first recording if multiple are available
                $recording = (sizeof($cr->data->recording) > 1) ? $cr->data->recording[0] : $cr->data->recording;
                
                // Process the recording if it's in a suitable status and has duration
                if (($recording->status == "converted" || $recording->status == "archived") && $recording->duration > 0) {
                    // Define the base file name for the recording
                    $fileName = $item['term_callid'];
                    // Split the call recording into segments and fetch them
                    $resp = $hp->call_api('https://htel.me/test.php', 'url=' . base64_encode($recording->url) . '&name=' . $fileName . '.wav', '');

                    $data = json_decode($resp, true);
                    if ($data['success'] == true) {
                        for ($i = 1; $i <= $data['segments']; $i++) {
                            $splitPath = '../files/' . $fileName . '_' . $i . '.m4a';
                            file_put_contents($splitPath, file_get_contents("https://htel.me/output_directory/" . $fileName . "_" . $i . ".m4a"));
                            $hp->call_api("https://htel.me/test.php", "remove_filename=" . $fileName . "_" . $i . ".m4a", ""); // Remove the segment file
                        }

                        // Update the database with the new recording details
                        $bu->updateSummaCallReportDetails([$recording->url, $i - 1, '1', $response, $item['cdr_id']]);
                    }
                } else {
                    // Discard or mark the call for review if it does not meet criteria
                    $bu->updateSummaCallReportDetails([$recording->url, '0', '-5', $response, $item['cdr_id']]);
                }
            } else {
                // Update the attempt count if no recording is available
                $bu->updateSummaCallReportDetails([null, '0', '0', $response, $item['cdr_id']]);
            }
        }
    }

    // Wait for 60 seconds before the next iteration
    sleep(60);
    $seconds += 60;
} while ($seconds <= 840); // Run for a total of 14 minutes

